/*
 * Copyright (c) 2010, Nieko Maatjes
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. This program may not be used for commercial purposes without specific
 *    prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY NIEKO MAATJES ''AS IS'' AND ANY
 * EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
 * WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL NIEKO MAATJES BE LIABLE FOR ANY
 * DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
 * (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
 * LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
 * ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
 * (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
 * SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 */

#include "main.h"

PSP_MODULE_INFO("PSP Tone Generator", PSP_MODULE_USER, 0, 2); // version: major, minor
PSP_MAIN_THREAD_ATTR(THREAD_ATTR_USER | THREAD_ATTR_VFPU);

// AUDIO
#define SAMPLE_RATE (44100)
#define SAMPLE_SIZE (1024)
static short buffer[SAMPLE_SIZE];
#define FREQUENCIES (3)
int curFrequency = 0, activeFrequencies = 0;
float curFrequencies[FREQUENCIES];
int enabledFrequencies[FREQUENCIES];
float posFrequencies[FREQUENCIES];
typedef struct { short l, r; } sample_t;

// CTRL
SceCtrlData pad, oldPad;

// EXTERN
extern unsigned int __attribute__((aligned(16))) guList[262144];

int main()
{
    setupExitCallback();
    graphicsInit();

    pspAudioInit();
    pspAudioSetChannelCallback(0, audioCallback, NULL);

    int i;
    for (i = 0; i < FREQUENCIES; i++)
    {
        curFrequencies[i] = 440.0f;
        enabledFrequencies[i] = 0;
        posFrequencies[i] = 0.0f;
    }

    while (1)
    {
        checkControls();

        // Start graphics
        sceGuStart(GU_DIRECT, guList);
        sceGuClearColor(0);
        sceGuClear(GU_COLOR_BUFFER_BIT);

        printInfo();

        // Audio callback calls tone generator

        displayWave(buffer);

        graphicsFlush();
    }

    die(NULL);

    return 0;
}

void checkControls()
{
    oldPad.Buttons = pad.Buttons;
    sceCtrlReadBufferPositive(&pad, 1);

    // Enable/disable tone
    if ((pad.Buttons ^ oldPad.Buttons)
     && (pad.Buttons & PSP_CTRL_CROSS))
    {
        enabledFrequencies[curFrequency] ^= 1;
        activeFrequencies += enabledFrequencies[curFrequency] == 1 ? 1 : -1;
    }

    // Change frequency to change
    if ((pad.Buttons ^ oldPad.Buttons)
     && (pad.Buttons & PSP_CTRL_UP))
    {
        curFrequency--;
        if (curFrequency < 0)
        { curFrequency = FREQUENCIES-1; }
    }
    else if ((pad.Buttons ^ oldPad.Buttons)
          && (pad.Buttons & PSP_CTRL_DOWN))
    {
        curFrequency++;
        if (curFrequency == FREQUENCIES)
        { curFrequency = 0; }
    }

    // Change current frequency
    int i;
    if (pad.Buttons & PSP_CTRL_RTRIGGER)
    {
        for (i = 0; i < FREQUENCIES; i++)
        {
            if ((i == curFrequency)
             || (pad.Buttons & PSP_CTRL_CIRCLE))
            { curFrequencies[i] += (pad.Buttons & PSP_CTRL_TRIANGLE) ? 20 : 1; }
        }
    }
    else if (pad.Buttons & PSP_CTRL_LTRIGGER)
    {
        for (i = 0; i < FREQUENCIES; i++)
        {
            if ((i == curFrequency)
             || (pad.Buttons & PSP_CTRL_CIRCLE))
            { curFrequencies[i] -= (pad.Buttons & PSP_CTRL_TRIANGLE) ? 20 : 1; }
        }
    }

    // Limit frequencies
    for (i = 0; i < FREQUENCIES; i++)
    {
        if (curFrequencies[i] < 20)    { curFrequencies[i] = 20; }
        if (curFrequencies[i] > 20480) { curFrequencies[i] = 20480; }
    }
}

void printInfo()
{
    pspDebugScreenSetXY(0, 0);
    pspDebugScreenSetTextColor(RGB(255, 255, 255));

    printf("X = enable/disable tone, L/R = change frequency\n");
    printf("O = change all frequencies, T = change frequency faster\n");

    int i;
    for (i = 0; i < FREQUENCIES; i++)
    {
        unsigned int color = enabledFrequencies[i] ? RGB(255, 255, 255) : RGB(127, 127, 127);
        pspDebugScreenSetTextColor(color);
        printf(" %s %d\n"
              ,i == curFrequency || (pad.Buttons & PSP_CTRL_CIRCLE)
               ? "->"
               : "  "
              ,(int)curFrequencies[i]
              );
    }
}

void toneGenerator(short *buffer)
{
    int i, j;
    for (i = 0; i < SAMPLE_SIZE; i++)
    {
        buffer[i] = 0;
        for (j = 0; j < FREQUENCIES; j++)
        {
            if (enabledFrequencies[j])
            {
                buffer[i] += (32767/activeFrequencies)*sinf(GU_PI*2.0f*curFrequencies[j]*posFrequencies[j]);
                posFrequencies[j] += 1.0f/SAMPLE_RATE;
            }
        }
    }

    double dummy;
    for (j = 0; j < FREQUENCIES; j++)
    {
        if (enabledFrequencies[j])
        {
            if (posFrequencies[j] * curFrequencies[j] >= 1.0f)
            { posFrequencies[j] = modf(posFrequencies[j] * curFrequencies[j], &dummy) / curFrequencies[j]; }
        }
    }
}

void displayWave(short *wave)
{
    float sx, sy, dx, dy;

    int i;

    for (i = 2; i < SAMPLE_SIZE; i += 2)
    {
        sx = ((float)i) * (SCREEN_WIDTH-1) / (SAMPLE_SIZE-1);
        sy = SCREEN_HEIGHT - (((float)wave[i])+32768) * (SCREEN_HEIGHT-1) / (65535);
        dx = ((float)i-2) * (SCREEN_WIDTH-1) / (SAMPLE_SIZE-1);
        dy = SCREEN_HEIGHT - (((float)wave[i-2])+32768) * (SCREEN_HEIGHT-1) / (65535);
        drawLine(sx, sy, dx, dy, 0xFFFFFFFF);
    }
}

// *buf = 16-bit stereo (short)
void audioCallback(void *buf, unsigned int length, void *userdata)
{
    toneGenerator(buffer);

    sample_t *out = (sample_t *)buf;
    int i;
    for (i = 0; i < SAMPLE_SIZE; i++)
    {
        out[i].l = buffer[i];
        out[i].r = buffer[i];
    }
}
