#include "SoundBuffer.h"

SoundBuffer *newSoundBuffer(unsigned int size, unsigned int frequency, char *note)
{
    SoundBuffer *new = malloc(sizeof(SoundBuffer));
    if (new == NULL)
    { die("Memory full!"); }

    new->size = size;
    new->frequency = frequency;
    new->decay = 0;
    if (frequency > 0)
    { new->decay = powf(0.995, 440.0f/frequency); }
    strncpy(new->note, note, 4);

    // Force the memory area to be contiguous (like an array)
    if ((new->samples = malloc(sizeof(ShortList)*size)) == NULL)
    { die("Memory full!"); }

    // Then force it to be a linked list anyway
    // (Makes it easier to move the entire array one position.)
    // Keep the array start address though, for free()'ing
    ShortList *current = new->sampleArray = new->samples;

    int i;
    for (i = 0; i < size-1; i++)
    {
        current[i].value = 0;
        current[i].next = &current[i+1];
    }

    // Close the loop
    current[size-1].value = 0;
    current[size-1].next = new->samples;

    return new;
}

void freeSoundBuffer(SoundBuffer *soundBuffer)
{
    if (soundBuffer != NULL)
    {
        // Free entire array (linked list) at once
        free(soundBuffer->sampleArray);
        soundBuffer->samples = soundBuffer->sampleArray = NULL;
        free(soundBuffer);
        soundBuffer = NULL;
    }
}

Chord *newChord()
{
    Chord *new = malloc(sizeof(Chord));
    if (new == NULL)
    { die("Memory full!"); }

    new->size = 0;

    return new;
}

SoundBuffer *addSoundBufferToChord(Chord *chord, unsigned int size, unsigned int frequency, char *note)
{
    SoundBufferList *current;

    // No more than 16 notes at once
    if (chord == NULL || chord->size >= 16)
    { return NULL; }

    if (chord->buffers == NULL)
    {
        if ((chord->buffers = malloc(sizeof(SoundBufferList))) == NULL)
        { die("Memory full!"); }

        current = chord->buffers;
    }
    else
    {
        current = chord->buffers;

        while (current->next != NULL)
        { current = current->next; }

        if ((current->next = malloc(sizeof(SoundBufferList))) == NULL)
        { die("Memory full!"); }

        current = current->next;
    }

    chord->size++;
    current->value = newSoundBuffer(size, frequency, note);
    current->next = NULL;

    return current->value;
}

void deleteSoundBufferFromChord(Chord *chord, SoundBuffer *soundBuffer)
{
    if (chord == NULL || chord->buffers == NULL)
    { return; }

    // First item, set new head
    if (chord->buffers->value == soundBuffer)
    {
        chord->buffers = chord->buffers->next;
        freeSoundBuffer(soundBuffer);
        chord->size--;

        return;
    }
    // Remember previous item to restore the broken chain
    else
    {
        int pos = 1;
        SoundBufferList *previous = chord->buffers;
        while (previous->next != NULL && previous->next->value != soundBuffer)
        { pos++; previous = previous->next; }

        if (previous->next == NULL)
        { return; }
        else
        {
            previous->next = previous->next->next;
            freeSoundBuffer(soundBuffer);
            chord->size--;
            return;
        }
    }
}

void freeChord(Chord *chord)
{
    if (chord != NULL)
    {
        int i = chord->size;
        while (i--)
        {
            SoundBufferList *toFree = chord->buffers;
            chord->buffers = chord->buffers->next;
            freeSoundBuffer(toFree->value);
        }
    }
}
